package com.vmware.samples.chassisui {

import com.vmware.actionsfw.ActionContext;
import com.vmware.actionsfw.events.ActionInvocationEvent;
import com.vmware.core.model.IResourceReference;
import com.vmware.core.model.StringDataObject;
import com.vmware.data.common.OperationType;
import com.vmware.data.common.events.ModelChangeEvent;
import com.vmware.data.query.events.PropertyRequest;
import com.vmware.flexutil.events.MethodReturnEvent;
import com.vmware.samples.chassisui.views.ChassisWizard;
import com.vmware.tiwo.TiwoFeatures;
import com.vmware.tiwo.events.TiwoRegistrationRequestEvent;
import com.vmware.widgets.util.Dialogs;

import flash.events.EventDispatcher;

import mx.controls.Alert;
import mx.events.CloseEvent;
[Event(name="{com.vmware.data.common.events.ModelChangeEvent.MODEL_CHANGE}",
      type="com.vmware.data.common.events.ModelChangeEvent")]
[Event(name="{com.vmware.tiwo.events.TiwoRegistrationRequestEvent.REQUEST_ID}",
      type="com.vmware.tiwo.events.TiwoRegistrationRequestEvent")]
[Event(name="{com.vmware.data.query.events.PropertyRequest.REQUEST_ID}",
      type="com.vmware.data.query.events.PropertyRequest")]
/**
* Command class to handle the chassis actions defined in plugin.xml.
*
* Implementation notes:
* - A [RequestHandler] method is defined for each ActionSpec uid.
* - Actions that trigger a server call use a common ChassisServiceProxy instance.
* - Proxy calls must handle the server's callbacks in the same class.
* - ModelChangeEvent notifies the plaftorm when an object is added, changed or deleted.
* - [EventHandler] methods should NOT be added here, see ChassisEventManager instead.
*/
public class ChassisActionCommand extends EventDispatcher {
   // Define a global proxy instance with the InjectableProxy tag
   [InjectableProxy]
   public var proxy:ChassisServiceProxy;

   [RequestHandler("com.vmware.samples.chassis.deleteChassis")]
   /**
    * Handles the delete chassis action invocation. This will remove the chassis from the
    * object store on the server after the user confirms.
    */
   public function onDeleteChassis(event:ActionInvocationEvent):void {
      const chassisRef:IResourceReference = getTargetObject(event);

      // Quick server call to retrieve the chassis name in order to display a better
	   // confirmation message. We could also avoid this and keep the message generic.
      dispatchEvent(PropertyRequest.newInstance(chassisRef, "name"));
   }

   [ResponseHandler(name="{com.vmware.data.query.events.PropertyRequest.RESPONSE_ID}")]
   public function onChassisNameRetrieved(request:PropertyRequest,
                                          result:StringDataObject, error:Error):void {
      if ((error != null) || (result == null)) {
         // For instance an ObjectNotFoundError may occur if another user deleted that chassis
         var errorMsg:String = (error != null ?
               error.message : "null result getting chassis name");
         Dialogs.showError(errorMsg, Util.getString("chassis.deleteAction"));
         return;
      }

      const chassisRef:IResourceReference = request.providerObject;
      const chassisName:String = result.value;

      var closeHandler:Function = function(event:CloseEvent):void {
         if (event.detail == Alert.NO) {
            return
         }
         // Service is called once user confirms
         proxy.deleteChassis(chassisRef, onDeleteChassisComplete, chassisRef);
      };

      var msg:String = Util.getString("chassis.confirmRemoveMsg", [chassisName]);
      Dialogs.showYesNoQuestion(
            msg, Util.getString("chassis.confirmRemove"), null, closeHandler);
   }

   [RequestHandler("com.vmware.samples.chassis.editChassis")]
   /**
    * Handles the edit chassis action invocation. This triggers a simple change of the
    * chassis type property to simulate some editing action.
    */
   public function onEditChassis(event:ActionInvocationEvent):void {
      var chassisRef:IResourceReference = getTargetObject(event);
      proxy.editChassis(chassisRef, onEditChassisComplete, chassisRef);
   }

   [RequestHandler("com.vmware.samples.chassis.createChassis")]
   /**
    * Handles the add chassis action. This opens the Create Chassis wizard.
    *
    * -------------------------------------------------------------------------------
    * IMPORTANT: As indicated in the SDK release notes Wizard and Dialogs APIs used
    * here are deprecated after 5.5. You can use them with 5.5 but you may have to
    * update your plugin before it can run with the next major release after 5.5.
    * -------------------------------------------------------------------------------
    */
   public function onCreateChassis(event:ActionInvocationEvent):void {
      var wizard:ChassisWizard = new ChassisWizard();
      var title:String = Util.getString("wizard.title");
      var icon:Class = Util.getImage("chassis");

      // It is best to display the wizard using the Tiwo Manager, i.e. register it with
      // a Tiwo event so that it can be minimized in the "Work In Progress" list
      // while the user is still working on it (Tiwo = Thing I am Working On)
      var tiwoEvent:TiwoRegistrationRequestEvent =
            new TiwoRegistrationRequestEvent(icon, title, wizard);

      // Add the persistence feature to allow active wizards to persist across user sessions.
      tiwoEvent.features.addFeature(TiwoFeatures.PERSISTENCE);

      // Remove the "help" feature which is set by default and points to vSphere help pages.
      // If you want to use the "?" button in the right-hand side of the wizard title bar
      // you can replace this line by:  wizard.helpURL = <your help site>;
      tiwoEvent.features.removeFeature(TiwoFeatures.HELP);

      // Dispatching the tiwoEvent opens the wizard. Note that the wizard result cannot
      // be handled in this class, the [EventHandler] for WizardCompletionEvent has to be
      // implemented in a global manager object because this ChassisActionCommand instance
      // must not be kept alive once the action is processed.  See ChassisEventManager.
      dispatchEvent(tiwoEvent);
   }

   [RequestHandler("com.vmware.samples.chassis.otherAction1")]
   [RequestHandler("com.vmware.samples.chassis.otherAction2")]
   [RequestHandler("com.vmware.samples.chassis.otherAction3")]
   public function onOtherAction(event:ActionInvocationEvent):void {
      // No implementation
   }

   private function onDeleteChassisComplete(event:MethodReturnEvent):void {
      if (event.error != null) {
         Alert.show(event.error.message);
         return;
      }

      // Use ModelChangeEvent with the modified object to update the UI
      var chassisRef:IResourceReference = event.callContext as IResourceReference;
      dispatchEvent(
         ModelChangeEvent.newSingleObjectChangeEvent(chassisRef,OperationType.DELETE));
   }

   private function onEditChassisComplete(event:MethodReturnEvent):void {
      if (event.error != null) {
         Alert.show(event.error.message);
         return;
      }

      // Use ModelChangeEvent with the modified object to update the UI
      var chassisRef:IResourceReference = event.callContext as IResourceReference;
      dispatchEvent(
         ModelChangeEvent.newSingleObjectChangeEvent(chassisRef,OperationType.CHANGE));
   }

   /**
    * Get the action target as a chassis reference
    *
    * @param event the ActionInvocationEvent
    * @return the chassis on which the action was called.
    */
   private function getTargetObject(event:ActionInvocationEvent):IResourceReference {
      // Throw an error if actionContext.targetObjects is not an array with 1 object
      // (ActionSpec property acceptsMultipleTargets is false by default)
      var actionContext:ActionContext = event.context;
      if (actionContext == null || (actionContext.targetObjects.length <= 0)
               || (!(actionContext.targetObjects[0] is IResourceReference))) {
         throw (new Error( "Invalid action context for " + event.type));
      }
      return (actionContext.targetObjects[0] as IResourceReference);
   }
}
}

